/*--------------------------------------------------
  Photometer Programm
  LED Photometer mit dem Lichtsensor TSL2561 oder BH1750,
  dreifarbiger LED (RGB-LED) oder einfarbige LED und
  mit eigenem WIFI AccessPoint und kleinem Webserver.

  HAW Hamburg - Labor BPA - Ulrich Scheffler
  Version 0.33
  06.10.2020
  --------------------------------------------------*/
const int led_farben_max = 3;    // Anzahl der LED Farben :=3 für eine RGB-LED  :=1 für eine einfarbige LED
#include <ESP8266WiFi.h>         // Bilbiothek für die ESP8266 Baugruppe und seine WIFI Funktionalität
#include <ESP8266WebServer.h>    // Bilbiothek für die ESP8266 Baugruppe und seine Web-Server Funktionalität
#include <Wire.h>                // Bilbiothek für die I2C Datenkummunikation - hier zum Datenaustausch mit dem Lichtsensor benötigt
#include <Adafruit_Sensor.h>     // Basis-Bilbiothek für die Adafruit Sensor Biblitheken (Bibliothek von Adafruit)
#include <Adafruit_TSL2561_U.h>  // Bibliothek für den Lichtsensor TSL2661 (Bibliothek von Adafruit)
#include <hp_BH1750.h>           // Bibliothek für den Lichtsensor BH1750 (Bibliotek von Stefan Armborst - MIT Lizenz)
#include "language_de.h"         // Sprachstrings mit einbinden

String sVersion        = "V0.33-";
String sVersion2       = " 06.10.2020 - HAW Hamburg, Lab. BPA, U. Scheffler";
const char* ssid       = "HAW-PHOTOMETER-"; // Der erste Teil des Namens des WIFI Access Points
const char* password   = "";                // Auf "" (leerer String) eingestellt für einen offenen Access Point ohne Passwort
unsigned long ulReqcount;                   // Zähler für die Aufrufe der Webseite
int ledPin[]              = {13, 12, 14};   // grüne LED an GPIO Pin 13/D7, rote LED an GPIO Pin 12/D6, blaue LED an GPIO Pin 14/D5 (hier für Lucky Light: LL-509RGBC2E-006)
const int sdaPin          = 4;              // SDA an GPIO/Pin  4 / D2   Anschluss-Pin für das SDA-Signal zur Datenkommunikation mit dem Lichtsensor
const int sclPin          = 5;              // SCL an GPIO/Pin  5 / D1   Anschluss-Pin für das SCL-Signal zur Datenkommunikation mit dem Lichtsensor
const int data_runs       = 7;              // Anzahl der Wiederholungsmessungen aus denen dann ein Mittelwert gebildet wird
float VIS_IRZEROdata[]    = {0.0, 0.0, 0.0};// Für die aktuellen Nullproben-Messwerte vom Lichtsensor.  Hier der Messwert vom Sensorteil, der im VIS und IR Bereich misst
float IRZEROdata[]        = {0.0, 0.0, 0.0};// Für die aktuellen Nullproben-Messwerte vom Lichtsensor.  Hier der Messwert vom Sensorteil, der nur im IR Bereich misst
float LUXZEROdata[]       = {0.0, 0.0, 0.0};// Für die aktuellen Nullproben-Messwerte vom Lichtsensor.  Hier die laut Datenblatt berechnete Beleuchtungsstärke in Lux
float VIS_IRdata[]        = {0.0, 0.0, 0.0};// Für die aktuellen Proben-Messwerte vom Lichtsensor.  Hier der Messwert vom Sensorteil, der im VIS und IR Bereich misst
float IRdata[]            = {0.0, 0.0, 0.0};// Für die aktuellen Proben-Messwerte vom Lichtsensor.  Hier der Messwert vom Sensorteil, der nur im IR Bereich misst
float LUXdata[]           = {0.0, 0.0, 0.0};// Für die aktuellen Proben-Messwerte vom Lichtsensor.  Hier die laut Datenblatt berechnete Beleuchtungsstärke in Lux
float VIS_IR_DARK_data[]  = {0.0, 0.0, 0.0};// Für die aktuellen Dunkel-Proben-Messwerte vom Lichtsensor.  Hier der Messwert vom Sensorteil, der im VIS und IR Bereich misst
float IR_DARK_data[]      = {0.0, 0.0, 0.0};// Für die aktuellen Dunkel-Proben-Messwerte vom Lichtsensor.  Hier der Messwert vom Sensorteil, der nur im IR Bereich misst
float LUX_DARK_data[]     = {0.0, 0.0, 0.0};// Für die aktuellen Dunkel-Proben-Messwerte vom Lichtsensor.  Hier die laut Datenblatt berechnete Beleuchtungsstärke in Lux
float E_LUX[]             = {0.0, 0.0, 0.0};// Für die berechnete Extiktionen aus den Beleuchtungsstärke-Daten
float E_VIS_IR[]          = {0.0, 0.0, 0.0};// Für die berechnete Extiktionen aus den Sensor-Daten vom Sensorteil der im VIS und IR Bereich misst
float E_IR[]              = {0.0, 0.0, 0.0};// Für die berechnete Extiktionen aus den Sensor-Daten vom Sensorteil der nur im IR Bereich misst
int probenzeilenIndex     = 0;              // Zeiger auf die aktuell zu füllende Probenzeile
const int probenzeilenMax = 5;              // Anzahl der Zeilen für Proben in der HTML Tabelle
float LUX_werte[3][probenzeilenMax];        // Array für Proben-Messwerte(Beleuchtungsstärke-Daten) vom Lichtsensor in Lux
float E_werte[3][probenzeilenMax];          // Array für berechnete Extiktionen aus den Beleuchtungsstärke-Daten
String anzeige            = "a";            // Kennung für die Anzeige der Tabellenanteile a=alle, g=grün, r=rot, b=blau
bool download             = false;          // Flag: Wenn True Download der DatenTabelle gewünscht, wenn False dan nicht.
String datentabelle       = "";             // Nimmt Datentabelle für den Download auf
const String trennzeichen = "\t";           // Spalten Trennzeichen für die Datentabelle (\t := Tabulatorzeichen)
uint16_t broadband        = 0;              // Sensor-Daten vom Sensorteil der im gesammten Messbereich des Sensors misst
uint16_t infrared         = 0;              // Sensor-Daten vom Sensorteil der nur im IR Bereich misst (falls vorhanden)
int sensortyp             = 0;              // Sensortyp:   0:= Kein Sensor; 1:= TSL2561; 2: BH1750

ESP8266WebServer server(80);                // Eine Instanz auf dem Server für Port 80 erzeugen
Adafruit_TSL2561_Unified tsl =Adafruit_TSL2561_Unified(TSL2561_ADDR_FLOAT, 12345);  //Objekt anlegen für den TSL2561 Sensor
hp_BH1750 BH1750;                           // Objekt anlegen für den BH1750 Sensor

bool readSensor(int color, bool LED_ON = true);  // Prototyp des Funktionsaufrufs anlegen

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
void displaySensorDetails(void) { // Zeigt einige Basisinformationen über den Sensor an
  Serial.println("------------------------------------");  
//  if (sensortyp==1){
//    sensor_t sensor;
//    tsl.getSensor(&sensor);
//    Serial.print  ("Sensor:       "); Serial.println(sensor.name);
//    //Serial.print  ("Driver Ver:   "); Serial.println(sensor.version);
//    //Serial.print  ("Unique ID:    "); Serial.println(sensor.sensor_id);
//    Serial.print  ("Max Value:    "); Serial.print(sensor.max_value); Serial.println(" lux");
//    Serial.print  ("Min Value:    "); Serial.print(sensor.min_value); Serial.println(" lux");
//    Serial.print  ("Resolution:   "); Serial.print(sensor.resolution); Serial.println(" lux");
//  }
  if (sensortyp==2){
    Serial.print  ("Sensor:       "); Serial.println("BH1750"); 
  }
  delay(500);
}

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
void configureSensor(void) { // Verstärkung und Integrationszeit konfigurieren
  if (sensortyp==1){
    // tsl.setGain(TSL2561_GAIN_1X);      // 1-fache Verstärkung      ... bei hoher Beleuchtungsstärke, um eine Sensor-Übersättigung zu verhindern
    // tsl.setGain(TSL2561_GAIN_16X);     // 16-fache Verstärkung     ... bei niediger Beleuchtungsstärke, um die Sensor-Empfindlichkeit zu erhöhen
    tsl.enableAutoRange(true);            // Automatische Verstärkung ... Verstärkung wechselt automatisch zwischen 1-fach und 16-fach
    // Das Ändern der Integrationszeit bewirkt eine Änderung der Sensor-Genauigkeit bzw. seiner Auflösung (402ms = 16-bit data)
    tsl.setIntegrationTime(TSL2561_INTEGRATIONTIME_13MS);      /* 13ms: Schnell aber niedrige Auflösung */
    //tsl.setIntegrationTime(TSL2561_INTEGRATIONTIME_101MS);   /* 101ms: Mittlere Geschwindigkeit und Auflösung */
    //tsl.setIntegrationTime(TSL2561_INTEGRATIONTIME_402MS);   /* 402ms: Langsamste Geschwindigkeit aber hohe Auflösung (16-bit Daten)*/
    // Einstellungen zur Info ausgeben
    Serial.print  ("Gain:         "); Serial.println("Auto");
    Serial.print  ("Timing:       "); Serial.println("13 ms");
    //Serial.print  ("Timing:       "); Serial.println("101 ms");
    Serial.println("------------------------------------");
  }
  if (sensortyp==2){
    //Serial.print  ("Sensor:       "); Serial.println("BH1750 set"); 
  }  
}

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
bool readSensor(int color, bool LED_ON) { // Sensor-Messdaten auslesen
  int ok = 0;                  // Zähler für geglückte Messungen
  float LUX[data_runs + 1];    // Daten Array zur Aufname der Einzelmessungen (ein Element mehr als Summenspeicher)
  float VIS_IR[data_runs + 1]; // Daten Array zur Aufnahme der Einzelmessungen
  float IR[data_runs + 1];     // Daten Array zur Aufnahme der Einzelmessungen
  float LUX_max = 0.0, LUX_min = 0.0;
  for (int j = 0; j <= data_runs; j++) { // Daten Arrays mit "0.0"-Werten vorbelegen
    LUX[j]    = 0.0;
    VIS_IR[j] = 0.0;
    IR[j]     = 0.0;
  }
  if (LED_ON){                // LED nicht einschalten zur Messung im "Dunkeln"
    if (led_farben_max > 1) {
      digitalWrite(ledPin[color], HIGH); // LED einschalten
    }
    else {
      digitalWrite(ledPin[1], HIGH); // LED einschalten
    }
  }
  for (int j = 0; j <= data_runs - 1; j++) { // Messungen "data_runs"-mal wiederholen und Einzelmessungen in Daten Array eintragen
    if (sensortyp==1){                    // TSL2561 Sensor
      sensors_event_t event;                              // Neues TSL2561 Lichtsensor Ereigniss einrichten
      tsl.getEvent(&event);                               // Messung durchführen
      if (event.light) {                                  // Wird nur dann "Wahr" wenn erfolgreich gemessen wurde
        LUX[j] = (event.light * 1.0);                     // LUX-Wert abholen
        if (j == 0) {LUX_max = LUX[j]; LUX_min = LUX[j];} // Min- und Max-Werte am Anfang auf den ersten Messwert setzten
        if (LUX[j] > LUX_max) LUX_max = LUX[j];           // Neuen Max-Wert suchen und ggf neu setzen
        if (LUX[j] < LUX_min) LUX_min = LUX[j];           // Neuen Min-Wert suchen und ggf neu setzen
        tsl.getLuminosity (&broadband, &infrared);        // VIS-IR- und IR-Werte auslesen
        VIS_IR[j] = (broadband * 1.0);                    // VIS-IR-Wert zuweisen
        IR[j]     = (infrared * 1.0);                     // IR-Wert zuweisen
        delay(25);                                        // Zwischen den Messungen warten (Zeit in Milli-Sekunden)
        ok += 1;                                          // Zähler für geglückte Messungen um 1 erhöhen
      }
      else { // Wenn "event.light = 0 lux" ist, dann ist der Sensor möglicher Weise auch gesättigt und es können keinen Daten generiert werden!
        Serial.println("Der Sensor-Messwert ist unbrauchbar. -> Sensor fehler!");
      }
    }
    if (sensortyp==2){                    // BH1750 Sensor
      BH1750.start(BH1750_QUALITY_HIGH2, 254);            // Startet eine Messung
      LUX[j] = (BH1750.getLux() * 1.0);                 // LUX-Wert abholen
      if (j == 0) {LUX_max = LUX[j]; LUX_min = LUX[j];} // Min- und Max-Werte am Anfang auf den ersten Messwert setzten
      if (LUX[j] > LUX_max) LUX_max = LUX[j];           // Neuen Max-Wert suchen und ggf neu setzen
      if (LUX[j] < LUX_min) LUX_min = LUX[j];           // Neuen Min-Wert suchen und ggf neu setzen
      delay(25);                                        // Zwischen den Messungen warten (Zeit in Milli-Sekunden)
      ok += 1;                                          // Zähler für geglückte Messungen um 1 erhöhen
    }
  }
  if (led_farben_max > 1) {
    digitalWrite(ledPin[color], LOW); // LED wieder ausschalten
  }
  else {
    digitalWrite(ledPin[1], LOW); // LED wieder ausschalten
  }
  if (ok >= data_runs) {                   // Nur wenn alle Einzelmessungen erfolgreich durchgeführt wurden ...
    for (int j = 0; j <= data_runs - 1; j++) { // Einzelmessungen aufaddieren
      LUX[data_runs]    += LUX[j];
      VIS_IR[data_runs] += VIS_IR[j];
      IR[data_runs]     += IR[j];
    }
    // Die aufaddierte Summe der Einzelwerte ohne den Max-Wert und den Min-Wert geteilt durch Anzahl der Einzelmessungen minus 2 ergibt den bereinigten Mittelwert
    LUX[data_runs]    = (LUX[data_runs] - LUX_max - LUX_min)    / (data_runs * 1.0 - 2.0);
    // Die aufaddierte Summe der Einzelwerte geteilt durch Anzahl der Einzelmessungen ergibt den jeweiligen Mittelwert
    VIS_IR[data_runs] = VIS_IR[data_runs] / (data_runs * 1.0);
    IR[data_runs]     = IR[data_runs]     / (data_runs * 1.0);
    LUXdata[color]    = LUX[data_runs];    // Berechneten LUX-Wert in die Datentabelle übertragen
    VIS_IRdata[color] = VIS_IR[data_runs]; // Berechneten VIS-IR-Wert in die Datentabelle übertragen
    IRdata[color]     = IR[data_runs];     // Berechneten IR-Wert in die Datentabelle übertragen
    return true;
  }
  else {
    return false;
  }
}

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
void cleardata() {
  Serial.println("Cleardata(RESET) aufgerufen!" );
  for (int zeilennummer = 0; zeilennummer < probenzeilenMax; zeilennummer++) { // Datentabelle mit "0.0"-Werten vorbelegen
    for (int ledfarbe = 0; ledfarbe < led_farben_max; ledfarbe++) {
      LUX_werte[ledfarbe][zeilennummer] = 0.0;
      E_werte[ledfarbe][zeilennummer]   = 0.0;
      LUXZEROdata[ledfarbe]             = 0.0;
      VIS_IRZEROdata[ledfarbe]          = 0.0;
      IRZEROdata[ledfarbe]              = 0.0;
    }
  }
  probenzeilenIndex = 0;
}

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
void setup() {
  // Globale Voreinstellungen
  for (int led = 0; led < led_farben_max; led++) {
    if (led_farben_max == 1)  pinMode(ledPin[1]  , OUTPUT);  // Wenn nur eine LED an GPIO Pin 12/D6 (für einfarbige LED), den Anschluss für die LED als Ausgang konfigurieren
    if (led_farben_max  > 1)  pinMode(ledPin[led], OUTPUT);  // Die Anschlüsse für die LEDs als Ausgänge konfigurieren
  }
  ulReqcount = 0;                      // Seitenaufrufszähler auf 0 setzten
  Serial.begin(115200);                // Serielle Verbindung initialisieren
  //WiFi.mode(WIFI_AP);                // WIFI Access Point Modus initialisieren
  
  if (led_farben_max > 1) {
    for (int led = 0; led < led_farben_max; led++) {
      digitalWrite(ledPin[led], HIGH); // LED einschalten
      delay(2000);                     // 2000 ms warten
      digitalWrite(ledPin[led], LOW);  // LED ausschalten
    }
  }
  else {
    digitalWrite(ledPin[1], HIGH); // LED einschalten
    delay(6000);                   // 6000 ms warten
    digitalWrite(ledPin[1], LOW);  // LED ausschalten
  }
  
  for (int zeilennummer = 0; zeilennummer < probenzeilenMax; zeilennummer++) { // Arrays mit "0.0"-Werten vorbelegen
    for (int ledfarbe = 0; ledfarbe < led_farben_max; ledfarbe++) {
      LUX_werte[ledfarbe][zeilennummer] = 0.0;
      E_werte[ledfarbe][zeilennummer]   = 0.0;
    }
  }
  
  Serial.println("");
  Serial.println(String(sVersion + String (led_farben_max) + sVersion2));
  // Das Anhängsel für den Access Point Namen (SSID) aus der MAC ID des ESP Moduls generieren.
  // Um den Namen nicht zu lang werden zu lassen, werden hier nur die letzten 2 Bytes verwendet.
  // Der Aufwand dient dazu, um einen möglichst einmaligen Access Point Namen für das jeweils verwendete ESP Modul entstehten zu lassen.
  uint8_t mac[WL_MAC_ADDR_LENGTH];
  WiFi.softAPmacAddress(mac);
  String macID = String((mac[WL_MAC_ADDR_LENGTH - 2] * 256 +  mac[WL_MAC_ADDR_LENGTH - 1]), DEC);
  macID.toUpperCase();
  String AP_NameString = ssid + macID;
  char AP_NameChar[AP_NameString.length() + 1];
  memset(AP_NameChar, 0, AP_NameString.length() + 1);
  for (int i = 0; i < AP_NameString.length(); i++) AP_NameChar[i] = AP_NameString.charAt(i);
  // WIFI Access Point starten, auf die entsprechende Funktion verweisen und den Web Server starten
  WiFi.softAP(AP_NameChar, password);
  server.on ( "/",handlePhotometer ); 
  server.onNotFound ( handlePhotometer );
  server.begin();
  
  Serial.print("WIFI Access Point started.   Name (SSID) : "); Serial.println(AP_NameChar);
  Serial.print("WEB-Server IP-Adress): "); Serial.println(WiFi.softAPIP());
  Serial.println("");
  // Lichtsensor TSL2561 oder BH1750
  Wire.pins(sdaPin, sclPin);  // Wire.pins(int sda, int scl) // Anschlusspins für das I2C-Protokoll zuordnen
  //if (tsl.begin()) {sensortyp= 1;}  // init the sensor, result (bool) wil be be "false" if no sensor found
  if (BH1750.begin(BH1750_TO_GROUND)){sensortyp=2;};// init the sensor with address pin connetcted to ground result (bool) wil be be "false" if no sensor found
  
  if (sensortyp==0)  {        // Falls kein Senor gefunden wurde eine Meldung ausgeben und LEDs blinken lassen
    Serial.print("Ooops, Light-Sensor not available ...! (Hardware Error) Program Stop!!! Restart necessary!!!");
    while (1){
        if (led_farben_max > 1) {
          for (int led = 0; led < led_farben_max; led++) {
            digitalWrite(ledPin[led], HIGH); // LED einschalten
            delay(200);                     // 200 ms warten
            digitalWrite(ledPin[led], LOW);  // LED ausschalten
          }
        }
        else {
          digitalWrite(ledPin[1], HIGH); // LED einschalten
          delay(200);                   // 200 ms warten
          digitalWrite(ledPin[1], LOW);  // LED ausschalten
        }
    }
  }
  displaySensorDetails();     // Zeigt einige Basisinformationen über den Sensor an
  configureSensor();          // Verstärkung und Integrationszeit konfigurieren

  if (led_farben_max > 1) {
    for (int led = 0; led < led_farben_max; led++) {
      digitalWrite(ledPin[led], HIGH); // LED einschalten
      delay(2000);                     // 2000 ms warten
      digitalWrite(ledPin[led], LOW);  // LED ausschalten
    }
  }
  else {
    digitalWrite(ledPin[1], HIGH); // LED einschalten
    delay(6000);                   // 6000 ms warten
    digitalWrite(ledPin[1], LOW);  // LED ausschalten
  }
  
  Serial.println("");
  handlePhotometer(); // Diese Zeile nur für Testzwecke aktivieren
}
//**************************************************************************************************************************************************************************************
void loop(){
  server.handleClient();
}

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
void handlePhotometer() {
  //String sRequest = "";
  String sCmd = "";
  if( server.method() == HTTP_GET ){
    if(server.args() >= 1){
      //sRequest = "GET /?" + server.argName ( 0 ) + "=" + server.arg ( 0 ) + "\n";
      //Serial.print("handlePhotometer: ");Serial.println(sRequest);
      sCmd = server.arg ( 0 );
      Serial.print("handlePhotometer: Command: "); Serial.println(sCmd);
    }
  }  
  // Die HTML Seite erzeugen
  String sResponseStart = "";
  String sResponseTab   = "";
  String sResponse      = "";

  if (sCmd.indexOf("READZERO") >= 0) {
    for (int color = 0; color < led_farben_max; color++) {
      if (!readSensor(color)) Serial.println("Sensor Error"); // Messung durchführen und Sensordaten auslesen - ggf. Fehler melden
      else {
        LUXZEROdata[color]    = LUXdata[color];
        VIS_IRZEROdata[color] = VIS_IRdata[color];
        IRZEROdata[color]     = IRdata[color];
      }
    }
  }
  if (sCmd.indexOf("READTSL") >= 0)  { // Wenn Button "Probe" gedrückt, ...
    for (int color = 0; color < led_farben_max; color++) {
      if (!readSensor(color)) Serial.println("Sensor Error"); // Messung durchführen und Sensordaten auslesen - ggf. Fehler melden
      LUX_werte[color][probenzeilenIndex] =  LUXdata[color];
      // Die Extinktionen berechnen
      if (LUXdata[color] > 0.0 & LUXZEROdata[color] > 0.0) {
        E_LUX[color] = -log10((LUXdata[color] * 1.0) / (LUXZEROdata[color] * 1.0));
      }
      else {
        E_LUX[color] = 0.0;
      }
      if (VIS_IRdata[color] > 0.0 & VIS_IRZEROdata[color] > 0.0) {
        E_VIS_IR[color] = -log10((VIS_IRdata[color] * 1.0) / (VIS_IRZEROdata[color] * 1.0));
      }
      else {
        E_VIS_IR[color] = 0.0;
      }
      if (IRdata[color] > 0.0 & IRZEROdata[color] > 0.0) {
        E_IR[color] = -log10((IRdata[color] * 1.0) / (IRZEROdata[color] * 1.0));
      }
      else {
        E_IR[color] = 0.0;
      }
    }
    probenzeilenIndex += 1;
    if (probenzeilenIndex >= probenzeilenMax) probenzeilenIndex = 0;
  }

  if (sCmd.indexOf("CLEARDATA") >= 0) {
    cleardata()    ; // Wenn Button "Reset" gedrückt, ...
  }
  if (led_farben_max > 1) {
    if (sCmd.indexOf("GR")       >= 0) {
      anzeige = "g"   ; // Wenn Button "grün" gedrückt, ...
    }
    if (sCmd.indexOf("RT")       >= 0) {
      anzeige = "r"   ; // Wenn Button "rot" gedrückt, ...
    }
    if (sCmd.indexOf("BL")       >= 0) {
      anzeige = "b"   ; // Wenn Button "blau" gedrückt, ...
    }
    if (sCmd.indexOf("ALL")      >= 0) {
      anzeige = "a"   ; // Wenn Button "alle" gedrückt, ...
    }
  }
  if (sCmd.indexOf("DOWNLOAD") >= 0) {
    download = true; // Wenn Button "Download" gedrückt, ...
  }
  // Die Extinktion mit ggf neuer Leerprobe erneut berechnen
  for (int color = 0; color < led_farben_max; color++) {
    for (int zeilennummer = 0; zeilennummer < probenzeilenMax; zeilennummer++) {
      if (LUX_werte[color][zeilennummer] > 0.0 & LUXZEROdata[color] > 0.0) {
        E_werte[color][zeilennummer] = -log10((LUX_werte[color][zeilennummer] * 1.0) / (LUXZEROdata[color] * 1.0));
      }
      else {
        E_werte[color][zeilennummer] = 0.0;
      }
    }
  }
  ulReqcount++;  // Seitenaufrufzähler um 1 raufzählen
  sResponseStart  = "<html><head><title>HAW Photometer</title></head>";
  sResponseStart += "<body bgcolor='#eFeFeF' style=\"font-family:'Arial'; color: #003CA0\">";  // Hintergrundfarbe, Schrift, Schriftfarbe
  sResponseStart += "<style>";
  sResponseStart += ".button {box-shadow: 2px 2px 0px 0px #003CA0;background:linear-gradient(to bottom,#ffffff 5%,#A0BEDC 100%);background-color:#ededed;border-radius:8px;border:1px solid #003CA0;display:inline-block;color:#003CA0;font-family:Arial;text-shadow:0px 1px 0px #ffffff;}";
  sResponseStart += ".button:hover {background:linear-gradient(to bottom, #A0BEDC 5%, #ffffff 100%);}";
  sResponseStart += ".button:active {position:relative;top:2px;}";
  sResponseStart += "table, th, td {border-collapse: collapse; }";
  sResponseStart += "</style>";
  
  sResponseStart += "<meta name=\"viewport\" content=\"width=device-width, initial-scale=1.0, user-scalable=yes\">";

  sResponseStart += "<p>";
  sResponseStart += "<div style=\"float:right;\">";
  sResponseStart += "<svg xmlns=\"http://www.w3.org/2000/svg\" xml:space=\"preserve\" width=\"27.1229mm\" height=\"7.2751mm\" viewBox=\"0 0 125324 33615\"  xmlns:xlink=\"http://www.w3.org/1999/xlink\">";
  sResponseStart += "<defs><style type=\"text/css\"><![CDATA[.fil0 {fill:#003CA0}.fnt0 {font-weight:bold;font-size:16300.1px;font-family:'Arial'}]]></style></defs>";
  sResponseStart += "<polygon class=\"fil0\" points=\"0,2484 24461,2484 24461,260 0,260 \"/>";
  sResponseStart += "<polygon class=\"fil0\" points=\"0,11378 24461,11378 24461,9154 0,9154 \"/>";
  sResponseStart += "<polygon class=\"fil0\" points=\"0,20273 24461,20273 24461,18049 0,18049 \"/>";
  sResponseStart += "<polygon class=\"fil0\" points=\"0,29168 24461,29168 24461,26945 0,26945 \"/>";
  sResponseStart += "<polygon class=\"fil0\" points=\"8896,6931 33357,6931 33357,4707 8896,4707 \"/>";
  sResponseStart += "<polygon class=\"fil0\" points=\"8896,15826 33357,15826 33357,13602 8896,13602 \"/>";
  sResponseStart += "<polygon class=\"fil0\" points=\"8896,24721 33357,24721 33357,22497 8896,22497 \"/>";
  sResponseStart += "<polygon class=\"fil0\" points=\"8896,33615 33357,33615 33357,31392 8896,31392 \"/>";
  sResponseStart += "<text x=\"41201\" y=\"11671\"  class=\"fil0 fnt0\">HAW</text><text x=\"41201\" y=\"30498\"  class=\"fil0 fnt0\">HAMBURG</text>";
  sResponseStart += "</svg>";
  sResponseStart += "</div>";
  sResponseStart += "<div style='text-align: left; font-size: 27; font-weight:bold;'>HAW Photometer</div>";
  sResponseStart += "</p>";
  
  sResponseStart += "<p style='text-align: center'><a href=\"?pin=READZERO\"><button class='button' style='font-size:26px;'>"; sResponseStart += TextLeerprobe; sResponseStart += "</button></a>";
  sResponseStart += String("&nbsp &nbsp &nbsp &nbsp <a href=\"?pin=READTSL\"><button class='button' style='font-size:26px;'>" + TextProbe + "&nbsp" + String(probenzeilenIndex + 1) + "</button></a></p>");
  sResponseStart += "<p style='text-align: center'><table border='0' style=\"font-family: 'Arial', sans-serif; color: #003CA0; margin-left: auto; margin-right: auto; font-size: 20 \">";
  for (int color = 0; color < led_farben_max; color++) {
    if (led_farben_max > 1) {
      sResponseStart += String("<tr><td>" + farbkennung[color] + "</td><td>:</td><td style='text-align: right'>" + String(LUXdata[color]) + " lx &nbsp</td><td style='text-align: right'>" + String(E_LUX[color], 4) + "</td></tr>");
    }
    else {
      sResponseStart += String("<tr><td>" + String(LUXdata[color]) + " lx &nbsp</td><td>" + String(E_LUX[color], 4) + "</td></tr>");
    }
  }
  sResponseStart += "</table></p>";
  sResponseStart += "<p style='text-align: center'>";
  sResponseStart += TabellenAuswahlText; sResponseStart += ":<BR>";
  if (led_farben_max > 1) {
    sResponseStart += String("<a href=\"?pin=GR\"><button class='button'>"  + farbkennung[0] + "</button></a>");
    sResponseStart += String("<a href=\"?pin=RT\"><button class='button'>"  + farbkennung[1] + "</button></a>");
    sResponseStart += String("<a href=\"?pin=BL\"><button class='button'>"  + farbkennung[2] + "</button></a>");
    sResponseStart += String("<a href=\"?pin=ALL\"><button class='button'>" + farbkennung[3] + "</button></a>");
  }
  sResponseStart += "</p>";
  if (led_farben_max > 1) {
    datentabelle  = String(" "            + trennzeichen + farbkennzeichung[0]        + trennzeichen + farbkennzeichung[0]    + trennzeichen + farbkennzeichung[1]        + trennzeichen + farbkennzeichung[1]    + trennzeichen + farbkennzeichung[2]        + trennzeichen + farbkennzeichung[2]    + "\r\n");
    datentabelle += String(" "            + trennzeichen + KennzeichungMesswert       + trennzeichen + KennzeichungExtinktion + trennzeichen + KennzeichungMesswert       + trennzeichen + KennzeichungExtinktion + trennzeichen + KennzeichungMesswert       + trennzeichen + KennzeichungExtinktion + "\r\n");
    datentabelle += String(" "            + trennzeichen + EinheitMesswert            + trennzeichen + EinheitExtinktion      + trennzeichen + EinheitMesswert            + trennzeichen + EinheitExtinktion      + trennzeichen + EinheitMesswert            + trennzeichen + EinheitExtinktion      + "\r\n");
    datentabelle += String(TextLeerprobe  + trennzeichen + String(LUXZEROdata[0], 2)  + trennzeichen + EinheitExtinktion      + trennzeichen + String(LUXZEROdata[1], 2)  + trennzeichen + EinheitExtinktion      + trennzeichen + String(LUXZEROdata[2], 2)  + trennzeichen + EinheitExtinktion      + "\r\n");
    for (int i = 0; i < probenzeilenMax; i++) {
      datentabelle += (TextProbe + " " + String(i + 1));
      datentabelle += (trennzeichen +  (String(LUX_werte[0][i], 2)) + trennzeichen + (String(String(E_werte[0][i], 3))));
      datentabelle += (trennzeichen +  (String(LUX_werte[1][i], 2)) + trennzeichen + (String(String(E_werte[1][i], 3))));
      datentabelle += (trennzeichen +  (String(LUX_werte[2][i], 2)) + trennzeichen + (String(String(E_werte[2][i], 3))));
      datentabelle += "\r\n";
    }
  }
  else {
    datentabelle =  String(" "            + trennzeichen + KennzeichungMesswert       + trennzeichen + KennzeichungExtinktion + "\r\n");
    datentabelle += String(" "            + trennzeichen + EinheitMesswert            + trennzeichen + EinheitExtinktion      + "\r\n");
    datentabelle += String(TextLeerprobe  + trennzeichen + String(LUXZEROdata[0], 2)  + trennzeichen + EinheitExtinktion      + "\r\n");
    for (int i = 0; i < probenzeilenMax; i++) {
      datentabelle += (TextProbe + " "  + String(i + 1));
      datentabelle += (trennzeichen +  (String(LUX_werte[0][i], 2)) + trennzeichen + (String(String(E_werte[0][i], 3))));
      datentabelle += "\r\n";
    }
  }
  Serial.println(datentabelle);
  
  if (led_farben_max == 1) {anzeige = "g";} // Wenn nur eine einfabige LED verbaut ist, wird immer nur die reduzierte Tabelle angezeigt
  sResponseTab += "<table border='1' width='100%' style=\"font-family: 'Arial', sans-serif; color:#003CA0;\"><tr style='text-align: center'>";
  sResponseTab += "<th rowspan='2'><a href=\"?pin=CLEARDATA\"> <button class='button'>Reset</button></a></th>";
  if (anzeige == "a" or anzeige == "g") sResponseTab += String("<th colspan='2' style='background-color: rgba(  0,255,  0, .2);'>"+farbkennzeichung[0]+"</th>");
  if (anzeige == "a" or anzeige == "r") sResponseTab += String("<th colspan='2' style='background-color: rgba(255,  0,  0, .2);'>"+farbkennzeichung[1]+"</th>");
  if (anzeige == "a" or anzeige == "b") sResponseTab += String("<th colspan='2' style='background-color: rgba(  0,  0,255, .2);'>"+farbkennzeichung[2]+"</th>");
  sResponseTab += "</tr><tr>";
  if (anzeige == "a" or anzeige == "g") sResponseTab += String("<th>"+TextMesswert+"<br>"+EinheitMesswert+"</th><th>"+TextExtinktion+"<br>"+EinheitExtinktion+"</th>");
  if (anzeige == "a" or anzeige == "r") sResponseTab += String("<th>"+TextMesswert+"<br>"+EinheitMesswert+"</th><th>"+TextExtinktion+"<br>"+EinheitExtinktion+"</th>");
  if (anzeige == "a" or anzeige == "b") sResponseTab += String("<th>"+TextMesswert+"<br>"+EinheitMesswert+"</th><th>"+TextExtinktion+"<br>"+EinheitExtinktion+"</th>");
  sResponseTab += "</tr>";
  
  
  sResponseTab += "<tr><td style='font-weight:bold;'>"; sResponseTab += TextLeerprobe; sResponseTab += "</td>";
  if (anzeige == "a" or anzeige == "g") {
    sResponseTab += String("<td style='text-align: center'>" + String(LUXZEROdata[0]) + "</td><td></td>");
  }
  if (anzeige == "a" or anzeige == "r") {
    sResponseTab += String("<td style='text-align: center'>" + String(LUXZEROdata[1]) + "</td><td></td>");
  }
  if (anzeige == "a" or anzeige == "b") {
    sResponseTab += String("<td style='text-align: center'>" + String(LUXZEROdata[2]) + "</td><td></td>");
  }
  sResponseTab += "</tr>";
  for (int i = 0; i < probenzeilenMax; i++) {
    sResponseTab += "<tr>";
    sResponseTab += String("<td style='font-weight:bold;'>" + TextProbe + "&nbsp" + String(i + 1) + "</td>");
    if (anzeige == "a" or anzeige == "g") {
      sResponseTab += String("<td style='text-align: center'>" + String(LUX_werte[0][i], 2) + "</td>");
      sResponseTab += String("<td style='text-align: center'>" + String(E_werte[0][i]  , 3) + "</td>");
    }
    if (anzeige == "a" or anzeige == "r") {
      sResponseTab += String("<td style='text-align: center'>" + String(LUX_werte[1][i], 2) + "</td>");
      sResponseTab += String("<td style='text-align: center'>" + String(E_werte[1][i]  , 3) + "</td>");
    }
    if (anzeige == "a" or anzeige == "b") {
      sResponseTab += String("<td style='text-align: center'>" + String(LUX_werte[2][i], 2) + "</td>");
      sResponseTab += String("<td style='text-align: center'>" + String(E_werte[2][i]  , 3) + "</td>");
    }
    sResponseTab += "</tr>";
  }
  sResponseTab += "</table>";
  sResponseTab += "<p style='text-align: center'><a href=\"?pin=DOWNLOAD\"><button class='button'>Download</button></a></p><BR>";
  sResponse += String("<FONT SIZE=-2>Seitenaufrufe: " + String(ulReqcount) + "<BR>"); // Aufrufzähler anzeigen
  sResponse += String(sVersion + String (led_farben_max) + sVersion2 + "<BR>");       // Versionshinweis anzeigen
  sResponse += "</body></html>";

  /* Die Antwort an den WIFI-Client senden */
  if (download) { // Wenn der Downloadbutton gedrückt wurde, soll eine Datei gestreamt werden ...
    download = false;
    sResponse += datentabelle;
    server.send ( 200, "text/csv", datentabelle );
    delay(1000);               // 1000 ms warten
  }
  else {  // Die HTML Seite ausgeben
    server.send ( 200, "text/html", sResponseStart + sResponseTab + sResponse);
  }
  Serial.println("handlePhotometer: Webpage done");
//  Serial.println("handlePhotometer: HTML Code start:");
//  Serial.println(sResponseStart);
//  delay(1000);
//  Serial.println(sResponseTab);
//  delay(1000);
//  Serial.println(sResponse);
//  Serial.println("handlePhotometer: HTML Code done");
  
}
